/***************************************************************************
 *
 * Copyright (c) 2020 Baidu.com, Inc. All Rights Reserved
 *
 **************************************************************************/

/**
 * @author Baidu
 * @brief demo
 *
 **/

#include <assert.h>
#include<dirent.h>
#include <iostream>
#include <sstream>
#include <sys/stat.h>

#include "easyedge/easyedge.h"
#include "easyedge/easyedge_config.h"
#include "easyedge/easyedge_gpu_turbo_config.h"

using namespace easyedge;

void list_file(std::string path, std::vector<std::string>& files) {
    DIR *p_dir;
    struct dirent *entry;
    if (p_dir = opendir(path.c_str())) {
        while (entry = readdir(p_dir)) {
            if (strcmp(entry->d_name, ".") != 0 && strcmp(entry->d_name, "..") != 0) {
                files.push_back(path + "/" + entry->d_name);
            }
        }
        closedir(p_dir);
    }
}

bool is_file(const std::string &path) {
    struct stat st;
    return stat(path.c_str(), &st) >= 0 && S_ISREG(st.st_mode);
}

int main(int argc, char *argv[]) {

    if (argc < 3) {
        std::cerr << "Usage: demo {model_dir} {image_name or image_directory} {serial_num}" << std::endl;
        exit(-1);
    }
    std::string model_dir = argv[1];
    std::string img_path = argv[2];
    std::string serial_num = "";

    if (argc >= 4) {
        serial_num = argv[3];
    }

    EdgeLogConfig log_config;
    log_config.enable_debug = false;
    log_config.to_file = false;
    global_controller()->set_log_config(log_config);

    EdgePredictorConfig config;
    config.model_dir = model_dir;
    config.set_config(params::PREDICTOR_KEY_SERIAL_NUM, serial_num);
    config.set_config(params::PREDICTOR_KEY_DEVICE_ID, 0);              // 设置需要使用的GPU
    config.set_config(params::PREDICTOR_KEY_GTURBO_MAX_BATCH_SIZE, 1);  // 优化的模型可以支持的最大batch_size，实际单次推理的图片数不能大于此值
    config.set_config(params::PREDICTOR_KEY_GTURBO_MAX_CONCURRENCY, 1); // 设置device对应的卡可以使用的最大线程数
    config.set_config(params::PREDICTOR_KEY_GTURBO_FP16, false);        // 置true开启fp16模式推理会更快，精度会略微降低，但取决于硬件是否支持fp16，不是所有模型都支持fp16，参阅文档
    config.set_config(params::PREDICTOR_KEY_GTURBO_COMPILE_LEVEL, 1);   // 编译模型的策略，如果当前设置的max_batch_size与历史编译存储的不同，则重新编译模型

    auto predictor = global_controller()->CreateEdgePredictor(config);

    if (predictor->init() != EDGE_OK) {
        exit(-1);
    }

    std::vector<cv::Mat> imgs;
    std::vector<std::string> img_files;
    std::vector<std::vector<EdgeResultData>> result;
    if (is_file(img_path)) {
        img_files.push_back(img_path);
        imgs.push_back(cv::imread(img_path));
    } else {
        list_file(img_path, img_files);
        // Make sure the number of images is not greater than max_batch_size you set
        assert(img_files.size() <= config.get_config<int>(params::PREDICTOR_KEY_GTURBO_MAX_BATCH_SIZE));
        for (auto& img : img_files) {
            imgs.push_back(cv::imread(img));
        }
    }

    predictor->infer(imgs, result);

    for (int i = 0; i < result.size(); ++i) {
        std::cout << "Results of image " << img_files[i] << ": " << std::endl;
        if (result[i].empty()) {
            std::cerr << "empty" << std::endl;
            continue;
        }
        for (auto &v : result[i]) {
            std::cout << v.index << ", " << v.label << ", p:" << v.prob;
            if (predictor->model_kind() == EdgeModelKind::kObjectDetection) {
                std::cout << " loc: "
                          << v.x1 << ", " << v.y1 << ", " << v.x2 << ", " << v.y2;
                auto p1 = cv::Point(static_cast<int>(v.x1 * imgs[i].cols),
                                    static_cast<int>(v.y1 * imgs[i].rows));
                auto p2 = cv::Point(static_cast<int>(v.x2 * imgs[i].cols),
                                    static_cast<int>(v.y2 * imgs[i].rows));
                cv::rectangle(imgs[i], p1, p2, RED, 2);
                cv::putText(imgs[i], v.label + " " +std::to_string(v.prob), p1, cv::FONT_HERSHEY_PLAIN, 1, BLUE, 2);
            }
            std::cout << std::endl;
        }
        // if (predictor->model_kind() == EdgeModelKind::kObjectDetection) {
        //     std::string result_image_path = img_files[i] + ".result.cpp.jpg";
        //     std::cout << "Write result to " << result_image_path << std::endl;
        //     cv::imwrite(result_image_path, imgs[i]);
        // }
    }

    std::cout << "Done" << std::endl;
    return 0;
}